package com.agiledeveloper;

import org.neo4j.graphdb.*;
import org.neo4j.graphdb.factory.GraphDatabaseFactory;
import org.neo4j.graphdb.traversal.*;
import org.neo4j.graphdb.traversal.Traverser;
import org.neo4j.kernel.Traversal;
import org.neo4j.kernel.impl.traversal.BidirectionalTraversalDescriptionImpl;

import java.util.HashMap;
import java.util.Map;

public class Neo4JSample {
  private final String DB_PATH = "/tmp/neo4j_db";
  private final String MAN = "man";
  private final String WOMAN = "woman";
  private final String LANGUAGE = "language";
  private final String NAME = "name";
  private GraphDatabaseService neo4jdb;
  private Map<String, Long> ids = new HashMap<>();

  private enum Relations implements RelationshipType {
    USES, LEARNING
  }

  public void run() {
    createDB();
    try (Transaction tx = neo4jdb.beginTx()) {
      createRelations();
      traverseRelations();
      tx.success();
    }
    shutdownDB();
  }

  private void traverseRelations() {
    Node john = neo4jdb.getNodeById(ids.get("John"));

    TraversalDescription evaluator =
      neo4jdb.traversalDescription()
        .relationships(Relations.USES)
        .relationships(Relations.LEARNING)
        .evaluator(Evaluators.atDepth(1))
        .evaluator(Evaluators.excludeStartPosition());

    Traverser traverser = evaluator.traverse(john);

    for (Path languagePath : traverser) {
      System.out.println("John " +
          languagePath.lastRelationship().getType().name() +  " " +
          languagePath.endNode().getProperty(NAME));
    }
  }

  private void createRelations() {
    createNode(MAN, "John");
    createNode(MAN, "Brad");
    createNode(MAN, "Joe");
    createNode(MAN, "Raju");
    createNode(WOMAN, "Sara");
    createNode(WOMAN, "Kim");
    createNode(WOMAN, "Paula");
    createNode(WOMAN, "Grace");
    createNode(LANGUAGE, "Java");
    createNode(LANGUAGE, "Groovy");
    createNode(LANGUAGE, "Scala");
    createNode(LANGUAGE, "Clojure");

    relate("John", Relations.USES, "Java");
    relate("John", Relations.LEARNING, "Scala");

    relate("Brad", Relations.USES, "Scala");
    relate("Brad", Relations.LEARNING, "Groovy");

    relate("Joe", Relations.USES, "Groovy");
    relate("Joe", Relations.LEARNING, "Clojure");

    relate("Raju", Relations.USES, "Clojure");
    relate("Raju", Relations.LEARNING, "Java");

    relate("Sara", Relations.USES, "Java");
    relate("Sara", Relations.LEARNING, "Scala");

    relate("Kim", Relations.USES, "Scala");
    relate("Kim", Relations.LEARNING, "Groovy");

    relate("Paula", Relations.USES, "Groovy");
    relate("Paula", Relations.LEARNING, "Clojure");

    relate("Grace", Relations.USES, "Clojure");
    relate("Grace", Relations.LEARNING, "Java");
  }

  private void relate(String personName, RelationshipType relationshipType, String languageName) {
    Node person = neo4jdb.getNodeById(ids.get(personName));
    Node language = neo4jdb.getNodeById(ids.get(languageName));

    person.createRelationshipTo(language, relationshipType);
  }

  public static void main(String[] args) {
    new Neo4JSample().run();
  }

  private void createNode(String label, String name) {
    Node node = neo4jdb.createNode(DynamicLabel.label(label));
    node.setProperty(NAME, name);
    ids.put(name, node.getId());
  }

  private void createDB() {
    neo4jdb = new GraphDatabaseFactory().newEmbeddedDatabase(DB_PATH);
  }

  private void shutdownDB() {
    neo4jdb.shutdown();
  }
}
