package com.agiledeveloper;

import junit.framework.TestCase;

import java.io.IOException;

import static org.easymock.EasyMock.*;

public class FileMonitorTest extends TestCase {
//  public void testAlarmIsRaised() throws InterruptedException {
//    FileSystemMock fileSystem = new FileSystemMock();
//    AlarmMock alarm = new AlarmMock();
//    FileMonitor monitor = new FileMonitor(fileSystem, alarm);
//
//    assertFalse(alarm.isRaised());
//
//    monitor.observe("C:\\temp");
//
//    fileSystem.simulateChange();
//    Thread.sleep(1000);
//    assertTrue(alarm.isRaised());
//  }
/*
  The above test is a failed attempt to test the observe method.
  (1) It is slow. Imagine 3000 tests, each taking a second long.
  (2) It is unreliable - there is a race condition on sleep.

  What's wrong?
  Test must be FAIR - Fast, Automated, Isolated, Repeatable.

  Unit test should be a test of a unit of code.
  What is a unit of code?
  A unit of code is smallest piece of code that does useful work.
  Getters and setters are smallest pieced of useless code. So, no need to
  explicitly unit test them.

  However, observe is useful, but not smallest piece of code. It can be
  refactored into smaller code.
 */

  public void testAlarmIsRaised() {
    FileSystemMock fileSystem = new FileSystemMock();
    AlarmMock alarm = new AlarmMock();
    FileMonitor monitor = new FileMonitor(fileSystem, alarm);

    monitor.setPathToObserve("C:\\temp");
    monitor.checkFileSystemAndRaiseAlarm();
    assertFalse(alarm.isRaised());

    fileSystem.simulateChange();

    monitor.checkFileSystemAndRaiseAlarm();
    assertTrue(alarm.isRaised());
  }

  //The key here is to keep the mocks isolated and trivial.
  public void testInvalidPath() {
    FileSystem fileSystem = new FileSystem() {
      public void setPathToObserve(String path) throws IOException {
        throw new IOException();
      }

      public boolean hasChanged() {
        return false;
      }
    };

    try {
      FileMonitor monitor = new FileMonitor(fileSystem, null);
      monitor.setPathToObserve("INVALID_PATH");
      fail("Expected exception for invalid path");
    } catch (FileSystemMonitorException ex) {
      //:) Success
    }
  }

  public void testAlarmIsRaisedUsingEasyMock() throws IOException {
    FileSystem fileSystem = createMock(FileSystem.class);
    Alarm alarm = createMock(Alarm.class);

    fileSystem.setPathToObserve("C:\\temp");
    expect(fileSystem.hasChanged()).andReturn(false);
    alarm.raise();
    expect(fileSystem.hasChanged()).andReturn(true);

    replay(fileSystem, alarm);

    FileMonitor monitor = new FileMonitor(fileSystem, alarm);
    monitor.setPathToObserve("C:\\temp");
    monitor.checkFileSystemAndRaiseAlarm();

    // simulate file change

    monitor.checkFileSystemAndRaiseAlarm();

    verify(fileSystem, alarm);
  }

  public void testInvaidPathUsingEasyMock() throws IOException {
    FileSystem fileSystem = createMock(FileSystem.class);

    fileSystem.setPathToObserve("INVALID_PATH");
    expectLastCall().andThrow(new IOException());

    replay(fileSystem);

    try {
      FileMonitor monitor = new FileMonitor(fileSystem, null);
      monitor.setPathToObserve("INVALID_PATH");
      fail("Expected exception for invalid path");
    } catch (FileSystemMonitorException ex) {
      //:) Success
    }

    verify(fileSystem);
  }
}
